<?php
// xo_memory.php
// API for getting/updating XO shared memory (schema + data).
// GET  = read memory (public)
// POST = full replace (console-only, API key required)

ini_set('display_errors', 0);
error_reporting(E_ALL);

// ===== CONFIG =====
// IMPORTANT: This must match the key used by xo_console.html and xo_manifest.php
$API_KEY = 'BQ0yO0wv9tQLvRUCPdmJAj1gYW1J27Zm';

require_once __DIR__ . '/xo_common.php';

// CORS / headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin', '*');
header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
header('Access-Control-Allow-Headers', 'Content-Type, X-XO-API-Key');

// Preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    echo json_encode(['ok' => true, 'mode' => 'options']);
    exit;
}

// ===== GET: read current memory (no auth required) =====
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $memory = xo_load_memory();

    $hash = hash('sha256', json_encode($memory));

    echo json_encode([
        'ok'          => true,
        'mode'        => 'get',
        'memory'      => $memory,
        'lastUpdated' => $memory['lastUpdatedAt'] ?? null,
        'sha256'      => $hash
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

// ===== POST: full replace (console-only, keyed) =====
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Require API key – prevents XO / arbitrary clients from full-memory writes
    $headerKey = $_SERVER['HTTP_X_XO_API_KEY'] ?? null;
    if ($headerKey !== $API_KEY) {
        xo_log('xo_memory_post', 'forbidden', [
            'reason' => 'invalid_or_missing_api_key'
        ]);
        http_response_code(403);
        echo json_encode([
            'ok'    => false,
            'mode'  => 'update',
            'error' => 'forbidden: invalid or missing API key for full memory update',
            'code'  => 'FORBIDDEN_FULL_UPDATE'
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        exit;
    }

    $raw = file_get_contents('php://input');
    $data = json_decode($raw, true);

    if (!is_array($data) || !isset($data['memory']) || !is_array($data['memory'])) {
        xo_log('xo_memory_post', 'error', [
            'reason' => 'invalid_payload',
            'body'   => $raw
        ]);
        http_response_code(400);
        echo json_encode([
            'ok'    => false,
            'mode'  => 'update',
            'error' => 'Invalid payload: expected { "memory": { ... } }',
            'code'  => 'INVALID_PAYLOAD'
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        exit;
    }

    $incoming = $data['memory'];

    // Ensure minimum structure; treat incoming as the authoritative full object
    if (!isset($incoming['data']) || !is_array($incoming['data'])) {
        $incoming['data'] = [];
    }
    foreach (['tasks','habits','reminders','projects'] as $key) {
        if (!isset($incoming['data'][$key]) || !is_array($incoming['data'][$key])) {
            $incoming['data'][$key] = [];
        }
    }

    if (!isset($incoming['schema']) || !is_array($incoming['schema'])) {
        // If console somehow omits schema, preserve existing one instead of nuking it.
        $existing = xo_load_memory();
        $incoming['schema'] = $existing['schema'] ?? [];
    }

    // Version is optional; if missing, keep existing version or default
    if (!isset($incoming['version'])) {
        $existing = isset($existing) ? $existing : xo_load_memory();
        $incoming['version'] = $existing['version'] ?? '0.1';
    }

    // Write entire memory object; actor is "console"
    xo_write_memory($incoming, 'console');

    $saved = xo_load_memory();
    $hash  = hash('sha256', json_encode($saved));

    xo_log('xo_memory_post', 'success', [
        'lastUpdatedAt' => $saved['lastUpdatedAt'] ?? null
    ]);

    echo json_encode([
        'ok'          => true,
        'mode'        => 'update',
        'memory'      => $saved,
        'lastUpdated' => $saved['lastUpdatedAt'] ?? null,
        'sha256'      => $hash
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

// Fallback: method not allowed
http_response_code(405);
echo json_encode([
    'error' => 'method_not_allowed'
], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
