<?php
// xo_manifest.php
// Simple API for getting/updating XO manifest text with NO auth.
// This restores the original, open behavior so XO can always read config.

// ===== CONFIG =====
$MANIFEST_PATH = __DIR__ . '/../xo_manifest.txt'; // adjust if needed

header('Content-Type: application/json; charset=utf-8');

// Basic CORS so browser/XO can talk to this easily
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

// Helper: load manifest text from file
function xo_load_manifest($path) {
    if (!file_exists($path)) {
        return [
            'ok'          => false,
            'mode'        => 'get',
            'manifest'    => '',
            'lastUpdated' => null,
            'sha256'      => null,
            'error'       => 'manifest_file_not_found'
        ];
    }
    $contents = file_get_contents($path);
    $mtime    = filemtime($path);
    $hash     = hash('sha256', $contents);

    return [
        'ok'          => true,
        'mode'        => 'get',
        'manifest'    => $contents,
        'lastUpdated' => $mtime ? date('c', $mtime) : null,
        'sha256'      => $hash
    ];
}

// GET  → read manifest
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $result = xo_load_manifest($MANIFEST_PATH);
    echo json_encode($result);
    exit;
}

// POST → update manifest text
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $raw = file_get_contents('php://input');
    $data = json_decode($raw, true);

    if (!is_array($data) || !isset($data['manifest']) || !is_string($data['manifest'])) {
        http_response_code(400);
        echo json_encode(['ok' => false, 'error' => 'invalid_payload']);
        exit;
    }

    $manifestText = $data['manifest'];

    if (file_put_contents($MANIFEST_PATH, $manifestText) === false) {
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'write_failed']);
        exit;
    }

    // Reload to return updated info
    $result = xo_load_manifest($MANIFEST_PATH);
    $result['mode'] = 'update';

    echo json_encode($result);
    exit;
}

// Anything else → method not allowed
http_response_code(405);
echo json_encode(['ok' => false, 'error' => 'method_not_allowed']);
