<?php
// xo_common.php
// Shared helpers for safe XO memory operations (read/write/log/priority).

ini_set('display_errors', 1); // <-- CRITICAL FIX: MUST BE 1 FOR TROUBLESHOOTING
error_reporting(E_ALL);

// Adjust paths if your layout differs.
$XO_MEMORY_PATH = __DIR__ . '/../xo_memory.json';
$XO_LOG_PATH    = __DIR__ . '/../xo_memory.log';

/**
 * Current ISO8601 UTC timestamp.
 */
function xo_now_iso(): string {
    return (new DateTimeImmutable('now', new DateTimeZone('UTC')))->format('c');
}

/**
 * Append a line of JSON log for auditing memory mutations.
 */
function xo_log(string $action, string $status, array $details = []): void {
    global $XO_LOG_PATH;

    $entry = [
        'timestamp' => xo_now_iso(),
        'action'    => $action,
        'status'    => $status,
        'details'   => $details
    ];

    @file_put_contents(
        $XO_LOG_PATH,
        json_encode($entry, JSON_UNESCAPED_SLASHES) . PHP_EOL,
        FILE_APPEND | LOCK_EX
    );
}

/**
 * JSON response helper.
 */
function xo_respond(array $data, int $statusCode = 200): void {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

/**
 * Load xo_memory.json, seeding a default structure if missing.
 */
function xo_load_memory(): array {
    global $XO_MEMORY_PATH;

    if (!file_exists($XO_MEMORY_PATH)) {
        // Seed with baseline structure.
        $seed = [
            "version" => "0.1",
            "schema"  => [
                "tasks" => [
                    "fields" => [
                        "id",
                        "title",
                        "status",
                        "impact",
                        "effort",
                        "dueDate",
                        "projectId",
                        "priorityScore",
                        "priorityLabel",
                        "createdAt",
                        "updatedAt",
                        "source",
                        "notes"
                    ]
                ],
                "habits" => [
                    "fields" => [
                        "id",
                        "name",
                        "schedule",
                        "active",
                        "createdAt",
                        "log"
                    ]
                ],
                "reminders" => [
                    "fields" => [
                        "id",
                        "name",
                        "type",
                        "date",
                        "time",
                        "recurrence",
                        "active",
                        "createdAt",
                        "notes"
                    ]
                ]
                ,
                "projects" => [
                    "fields" => [
                        "id",
                        "name",
                        "dueDate"
                    ]
                ]
            ],
            "data" => [
                "tasks"     => [],
                "habits"    => [],
                "reminders" => [],
                "projects"  => []
            ],
            "lastUpdatedAt" => xo_now_iso(),
            "lastUpdatedBy" => "server-init"
        ];

        xo_write_memory($seed, 'server-init');
        return $seed;
    }

    $raw = file_get_contents($XO_MEMORY_PATH);
    if ($raw === false) {
        xo_log('load_memory', 'error', ['reason' => 'file_get_contents_failed']);
        xo_respond(['success' => false, 'error' => 'Unable to read memory file'], 500);
    }

    $json = json_decode($raw, true);
    if (!is_array($json)) {
        xo_log('load_memory', 'error', ['reason' => 'json_decode_failed']);
        xo_respond(['success' => false, 'error' => 'Memory file is not valid JSON'], 500);
    }

    // Ensure core data arrays exist.
    if (!isset($json['data']) || !is_array($json['data'])) {
        $json['data'] = [];
    }

    foreach (['tasks','habits','reminders','projects'] as $key) {
        if (!isset($json['data'][$key]) || !is_array($json['data'][$key])) {
            $json['data'][$key] = [];
        }
    }

    return $json;
}

/**
 * Atomically write xo_memory.json with updated metadata.
 */
function xo_write_memory(array $memory, string $actor): void {
    global $XO_MEMORY_PATH;

    $memory['lastUpdatedAt'] = xo_now_iso();
    $memory['lastUpdatedBy'] = $actor;

    $json = json_encode($memory, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if ($json === false) {
        xo_log('write_memory', 'error', ['reason' => 'json_encode_failed']);
        xo_respond(['success' => false, 'error' => 'Failed to encode memory JSON'], 500);
    }

    $dir = dirname($XO_MEMORY_PATH);
    if (!is_dir($dir)) {
        mkdir($dir, 0775, true);
    }

    $tmp = $XO_MEMORY_PATH . '.tmp';
    $bytes = file_put_contents($tmp, $json, LOCK_EX);
    if ($bytes === false) {
        xo_log('write_memory', 'error', ['reason' => 'temp_write_failed']);
        xo_respond(['success' => false, 'error' => 'Failed to write temp memory file'], 500);
    }

    if (!rename($tmp, $XO_MEMORY_PATH)) {
        xo_log('write_memory', 'error', ['reason' => 'rename_failed']);
        xo_respond(['success' => false, 'error' => 'Failed to commit memory file'], 500);
    }

    xo_log('write_memory', 'success', ['actor' => $actor]);
}

/**
 * Compute the next integer ID for an array of items with 'id' fields.
 */
function xo_next_id(array $items): int {
    $max = 0;
    foreach ($items as $item) {
        if (isset($item['id']) && is_int($item['id']) && $item['id'] > $max) {
            $max = $item['id'];
        }
    }
    return $max + 1;
}

/**
 * Calculate task priorityScore and priorityLabel from impact, effort, and dueDate.
 *
 * impact: 1–5 (required for new tasks)
 * effort: 1–5 (required for new tasks)
 * dueDate: YYYY-MM-DD or null
 *
 * Returns: [float|null $score, string|null $label]
 */
function xo_calculate_task_priority(?int $impact, ?int $effort, ?string $dueDate): array {
    // If we don't have enough info, return unknown priority.
    if ($impact === null || $effort === null || $impact <= 0 || $effort <= 0) {
        return [null, null];
    }

    // Compute days to due from today (UTC).
    $urgency = 1; // default low

    if (!empty($dueDate)) {
        try {
            $tz    = new DateTimeZone('UTC');
            $today = new DateTimeImmutable('today', $tz);
            $due   = new DateTimeImmutable($dueDate, $tz);

            $diffDays = (int)$today->diff($due)->format('%r%a'); // signed days

            if ($diffDays <= 0) {
                $urgency = 5; // overdue or due today
            } elseif ($diffDays <= 2) {
                $urgency = 5;
            } elseif ($diffDays <= 4) {
                $urgency = 4;
            } elseif ($diffDays <= 7) {
                $urgency = 3;
            } elseif ($diffDays <= 14) {
                $urgency = 2;
            } else {
                $urgency = 1;
            }
        } catch (Exception $e) {
            // Invalid date string → treat as low urgency
            $urgency = 1;
        }
    } else {
        // No due date given → treat as low urgency by default
        $urgency = 1;
    }

    // Score: (impact × urgency) ÷ effort
    $score = ($impact * $urgency) / max($effort, 1);
    $scoreRounded = round($score, 1);

    // Map score to label
    if ($scoreRounded >= 8.0) {
        $label = 'Critical';
    } elseif ($scoreRounded >= 5.0) {
        $label = 'High';
    } elseif ($scoreRounded >= 2.5) {
        $label = 'Medium';
    } else {
        $label = 'Low';
    }

    return [$scoreRounded, $label];
}