<?php
// update_task.php
// Safely update an existing task in xo_memory.json by id, patching only allowed fields.
// Recomputes priorityScore/priorityLabel when impact/effort/dueDate change.

ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/xo_common.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    xo_respond(['success' => false, 'error' => 'Method not allowed'], 405);
}

// Parse JSON body
$raw   = file_get_contents('php://input');
$input = json_decode($raw, true);

if (!is_array($input)) {
    xo_log('update_task', 'error', ['reason' => 'invalid_json', 'body' => $raw]);
    xo_respond(['success' => false, 'error' => 'Invalid JSON body'], 400);
}

// Required: id
$id = $input['id'] ?? null;
if (!is_int($id)) {
    // Allow numeric strings but convert
    if (is_string($id) && ctype_digit($id)) {
        $id = (int) $id;
    } else {
        xo_log('update_task', 'error', ['reason' => 'missing_or_invalid_id', 'input' => $input]);
        xo_respond([
            'success' => false,
            'error'   => 'Missing or invalid required field: id',
            'code'    => 'VALIDATION_ERROR'
        ], 400);
    }
}

// Optional fields to patch
$fieldsToPatch = [];
$allowed = ['title', 'status', 'impact', 'effort', 'dueDate', 'notes', 'projectId'];

// Normalize & validate impact/effort if present
if (array_key_exists('impact', $input)) {
    $impact = $input['impact'];
    if (is_string($impact) && ctype_digit($impact)) {
        $impact = (int)$impact;
    }
    if (!is_int($impact) || $impact < 1 || $impact > 5) {
        xo_log('update_task', 'error', ['reason' => 'invalid_impact', 'input' => $input]);
        xo_respond([
            'success' => false,
            'error'   => 'Invalid field: impact (expected integer 1–5)',
            'code'    => 'VALIDATION_ERROR'
        ], 400);
    }
    $fieldsToPatch['impact'] = $impact;
}

if (array_key_exists('effort', $input)) {
    $effort = $input['effort'];
    if (is_string($effort) && ctype_digit($effort)) {
        $effort = (int)$effort;
    }
    if (!is_int($effort) || $effort < 1 || $effort > 5) {
        xo_log('update_task', 'error', ['reason' => 'invalid_effort', 'input' => $input]);
        xo_respond([
            'success' => false,
            'error'   => 'Invalid field: effort (expected integer 1–5)',
            'code'    => 'VALIDATION_ERROR'
        ], 400);
    }
    $fieldsToPatch['effort'] = $effort;
}

// dueDate can be any non-empty string; validation happens in priority helper
if (array_key_exists('dueDate', $input)) {
    $fieldsToPatch['dueDate'] = $input['dueDate'];
}

// title, status, notes patch through as-is (trim strings)
foreach (['title','status','notes'] as $field) {
    if (array_key_exists($field, $input)) {
        $value = $input[$field];
        if (is_string($value)) {
            $value = trim($value);
        }
        $fieldsToPatch[$field] = $value;
    }
}

// projectId normalization
if (array_key_exists('projectId', $input)) {
    $projectId = $input['projectId'];
    if ($projectId !== null) {
        if (is_string($projectId) && ctype_digit($projectId)) {
            $projectId = (int)$projectId;
        }
        if (!is_int($projectId)) {
            xo_log('update_task', 'warn', ['reason' => 'invalid_projectId_type', 'value' => $projectId]);
            $projectId = null;
        }
    }
    $fieldsToPatch['projectId'] = $projectId;
}

if (empty($fieldsToPatch)) {
    xo_log('update_task', 'error', ['reason' => 'no_fields_to_patch', 'input' => $input]);
    xo_respond([
        'success' => false,
        'error'   => 'No updatable fields provided',
        'code'    => 'NO_FIELDS'
    ], 400);
}

$source = $input['source'] ?? 'XO';
$actor  = $source ?: 'XO';

// 1) Load existing memory
$memory = xo_load_memory();

if (!isset($memory['data']['tasks']) || !is_array($memory['data']['tasks'])) {
    $memory['data']['tasks'] = [];
}

// 2) Find the task by id
$foundIndex = null;
foreach ($memory['data']['tasks'] as $idx => $task) {
    if (isset($task['id']) && (int)$task['id'] === $id) {
        $foundIndex = $idx;
        break;
    }
}

if ($foundIndex === null) {
    xo_log('update_task', 'error', ['reason' => 'task_not_found', 'id' => $id]);
    xo_respond([
        'success' => false,
        'error'   => 'Task not found',
        'code'    => 'NOT_FOUND',
        'id'      => $id
    ], 404);
}

// 3) Patch the task (only allowed fields)
$task = $memory['data']['tasks'][$foundIndex];

foreach ($fieldsToPatch as $field => $value) {
    $task[$field] = $value;
}

// 4) Recompute priority based on current impact/effort/dueDate
$currentImpact = $task['impact']  ?? null;
$currentEffort = $task['effort']  ?? null;
$currentDue    = $task['dueDate'] ?? null;

list($priorityScore, $priorityLabel) = xo_calculate_task_priority(
    is_int($currentImpact) ? $currentImpact : null,
    is_int($currentEffort) ? $currentEffort : null,
    is_string($currentDue) ? $currentDue : null
);

$task['priorityScore'] = $priorityScore;
$task['priorityLabel'] = $priorityLabel;

// Always update updatedAt
$task['updatedAt'] = xo_now_iso();

// Write patched task back into array
$memory['data']['tasks'][$foundIndex] = $task;

// 5) Persist safely
xo_write_memory($memory, $actor);

// 6) Log and respond
xo_log('update_task', 'success', [
    'id'            => $id,
    'patched_fields'=> array_keys($fieldsToPatch),
    'priorityScore' => $priorityScore,
    'priorityLabel' => $priorityLabel
]);

xo_respond([
    'success'       => true,
    'action'        => 'update_task',
    'task'          => $task,
    'lastUpdatedAt' => $memory['lastUpdatedAt'] ?? null
]);
