<?php
// update_project.php
// Safely update an existing project in xo_memory.json by id, patching only allowed fields.

ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/xo_common.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    xo_respond(['success' => false, 'error' => 'Method not allowed'], 405);
}

// Parse JSON body
$raw   = file_get_contents('php://input');
$input = json_decode($raw, true);

if (!is_array($input)) {
    xo_log('update_project', 'error', ['reason' => 'invalid_json', 'body' => $raw]);
    xo_respond(['success' => false, 'error' => 'Invalid JSON body'], 400);
}

// Required: id
$id = $input['id'] ?? null;
if (!is_int($id)) {
    // Allow numeric strings but convert
    if (is_string($id) && ctype_digit($id)) {
        $id = (int) $id;
    } else {
        xo_log('update_project', 'error', ['reason' => 'missing_or_invalid_id', 'input' => $input]);
        xo_respond([
            'success' => false,
            'error'   => 'Missing or invalid required field: id',
            'code'    => 'VALIDATION_ERROR'
        ], 400);
    }
}

// Optional fields to patch (only these are allowed)
$fieldsToPatch = [];
$allowed = ['name', 'dueDate'];

// Only include fields actually present in the payload
foreach ($allowed as $field) {
    if (array_key_exists($field, $input)) {
        $fieldsToPatch[$field] = $input[$field];
    }
}

if (empty($fieldsToPatch)) {
    xo_log('update_project', 'error', ['reason' => 'no_fields_to_patch', 'input' => $input]);
    xo_respond([
        'success' => false,
        'error'   => 'No updatable fields provided',
        'code'    => 'NO_FIELDS'
    ], 400);
}

$source = $input['source'] ?? 'XO';
$actor  = $source ?: 'XO';

// 1) Load existing memory
$memory = xo_load_memory();

if (!isset($memory['data']['projects']) || !is_array($memory['data']['projects'])) {
    $memory['data']['projects'] = [];
}

// 2) Find the project by id
$foundIndex = null;
foreach ($memory['data']['projects'] as $idx => $project) {
    if (isset($project['id']) && (int)$project['id'] === $id) {
        $foundIndex = $idx;
        break;
    }
}

if ($foundIndex === null) {
    xo_log('update_project', 'error', ['reason' => 'project_not_found', 'id' => $id]);
    xo_respond([
        'success' => false,
        'error'   => 'Project not found',
        'code'    => 'NOT_FOUND',
        'id'      => $id
    ], 404);
}

// 3) Patch the project (only allowed fields)
$project = $memory['data']['projects'][$foundIndex];

foreach ($fieldsToPatch as $field => $value) {
    if (is_string($value)) {
        $value = trim($value);
    }
    $project[$field] = $value;
}

// NOTE: Projects currently don't track createdAt/updatedAt in schema.
// If you later add those fields to schema, you can also set updatedAt here.

// Write patched project back into array
$memory['data']['projects'][$foundIndex] = $project;

// 4) Persist safely
xo_write_memory($memory, $actor);

// 5) Log and respond
xo_log('update_project', 'success', ['id' => $id, 'patched_fields' => array_keys($fieldsToPatch)]);

xo_respond([
    'success'       => true,
    'action'        => 'update_project',
    'project'       => $project,
    'lastUpdatedAt' => $memory['lastUpdatedAt'] ?? null
]);
