<?php
// /api/search_memory.php (REVISED FILE)
// Endpoint for the Gemini Bridge to search memory items by category and query.

require_once __DIR__ . '/xo_common.php';

// ===== HTTP / CORS HEADERS (For external calls, though primarily used internally) =====
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { http_response_code(405); exit; }

// --- Get Data from Request Body (Dual-purpose input fix) ---
global $XO_PAYLOAD;

if (isset($XO_PAYLOAD) && is_array($XO_PAYLOAD)) {
    // 1. Running as a tool call from chat.php
    $data = $XO_PAYLOAD;
} else {
    // 2. Running as a standard HTTP POST from the browser
    $raw  = file_get_contents('php://input');
    $data = json_decode($raw, true);
}

$category = $data['category'] ?? null;
$query    = $data['query'] ?? null;

if (empty($category) || empty($query)) {
    http_response_code(400);
    xo_respond(['success' => false, 'error' => 'invalid_payload', 'details' => 'Missing category or query.']);
}

$memory = xo_load_memory();
$results = [];

// 1. Basic Validation
if (!isset($memory['schema'][$category])) {
    // Respond successfully, but with an empty list and a note, so the AI can report back cleanly.
    xo_respond([
        'success' => true,
        'category' => $category,
        'results' => [],
        'note' => 'Category not found in memory schema.'
    ]);
}

// Ensure the category data exists
$items = $memory['data'][$category] ?? [];

// --- 2. Implement Simple Search Logic ---

$query = strtolower(trim($query));
$query_parts = explode(':', $query, 2);

if (count($query_parts) === 2) {
    // Case 1: Key:Value Search (e.g., "status: open" or "id: 3")
    $search_key = trim($query_parts[0]);
    $search_value = trim($query_parts[1]);

    // Handle "id: N" specifically for direct lookup
    if ($search_key === 'id' && is_numeric($search_value)) {
        $id = (int)$search_value;
        foreach ($items as $item) {
            if (isset($item['id']) && (int)$item['id'] === $id) {
                $results[] = $item;
                break; // Stop after finding the specific ID
            }
        }
    } else {
        // Generic key/value match (case-insensitive on string values)
        foreach ($items as $item) {
            if (isset($item[$search_key])) {
                // If item value is a string and contains the search value
                if (is_string($item[$search_key]) && stripos($item[$search_key], $search_value) !== false) {
                    $results[] = $item;
                }
                // If item value is numeric and matches the search value
                elseif (is_numeric($item[$search_key]) && (string)$item[$search_key] === $search_value) {
                    $results[] = $item;
                }
            }
        }
    }
} else {
    // Case 2: Free Text Search (search against title/name/notes)
    foreach ($items as $item) {
        $match = false;
        // Search primary identifier (title or name)
        $identifier = $item['title'] ?? $item['name'] ?? null;
        if ($identifier && stripos($identifier, $query) !== false) {
            $match = true;
        }
        // Search notes/description
        elseif (isset($item['notes']) && is_string($item['notes']) && stripos($item['notes'], $query) !== false) {
            $match = true;
        }
        
        if ($match) {
            $results[] = $item;
        }
    }
}

// --- 3. Return the Search Results ---
xo_log('search_memory', 'success', ['category' => $category, 'query' => $query, 'count' => count($results)]);

// Respond with the list of matching items
xo_respond([
    'success'  => true,
    'category' => $category,
    'query'    => $query,
    'results'  => $results
]);