<?php
// chat.php
// Gemini Bridge for XO Orchestration System

// Enable error display for debugging purposes (MUST BE 1 for troubleshooting)
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Load the common functions (memory I/O, logging, priority calculation)
require_once 'xo_common.php';

// Load Composer dependencies (must be installed via 'composer require google/apiclient')
require __DIR__ . '/../vendor/autoload.php';

// Define your Gemini API Key here (USER'S KEY INSERTED)
$GEMINI_API_KEY = 'AIzaSyCBMm1_OmsfZQA_NpDC1vUjlC6V-HCbYMg'; 

// --- Setup ---

if (empty($GEMINI_API_KEY) || $GEMINI_API_KEY === 'YOUR_API_KEY_HERE') {
    xo_log('chat', 'error', ['reason' => 'API_KEY_MISSING']);
    xo_respond(['success' => false, 'error' => 'Gemini API Key is not configured in chat.php'], 500);
}

// 1. Initialize the Google Client
$client = new Google\Client();
$client->setDeveloperKey($GEMINI_API_KEY);

// 2. Determine the model to use
$model = 'gemini-2.5-flash';

// 3. Initialize the Generative Model Service
// Note: We are using the correct class for the Google API Client library
$service = new Google\Service\AiPlatform\GenerativeModelService($client);

// 4. Configure the model request
$request = new Google\Service\AiPlatform\GenerateContentRequest();
$request->setModel($model);
$request->setConfig(new Google\Service\AiPlatform\GenerateContentConfig([
    'temperature' => 0.2,
    'systemInstruction' => "You are XO, a helpful and efficient orchestration assistant. Your primary goal is to manage the user's personal memory data (tasks, habits, reminders, projects) stored in JSON.
    When a user asks to add, remove, or modify data, you MUST respond with a JSON object containing the action and the data to be mutated. You MUST NOT include any conversational text outside of the JSON block.
    
    The user's current memory state is passed in the JSON body. Analyze the user's request against the memory state to determine the appropriate response.
    
    If the user's request is purely conversational (e.g., 'Hello', 'How are you?', 'What is the capital of DC?'), you MUST respond with a JSON object that has:
    - 'action': 'respond'
    - 'response': [your friendly, conversational reply].
    
    The DC rule is in effect: DC is DC, not District of Columbia. The user's preferences are: Carrick is deathly afraid of the dark. Keep this in mind for conversational responses.
    
    Available actions for memory manipulation:
    - 'action': 'add_task', 'data': [{...task_fields...}]
    - 'action': 'update_task', 'data': [{...task_fields_to_update...}]
    - 'action': 'delete_task', 'data': [{'id': 1}, {'id': 5}]
    - 'action': 'add_habit', 'data': [{...habit_fields...}]
    - ...and so on for reminders and projects.
    
    When adding a task, you MUST calculate the 'impact' (1-5) and 'effort' (1-5) based on the context, and set 'status' to 'New'.
    
    Example response for adding a task:
    {
      \"action\": \"add_task\",
      \"data\": [
        {
          \"title\": \"Check the tires on the DC car\",
          \"impact\": 4,
          \"effort\": 2,
          \"dueDate\": \"2025-12-06\",
          \"status\": \"New\",
          \"source\": \"Gemini\"
        }
      ]
    }",
]));

// --- Execution ---

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    xo_respond(['success' => false, 'error' => 'Only POST requests are accepted'], 405);
}

// Load the incoming request data
$input = file_get_contents('php://input');
$requestData = json_decode($input, true);

if (!isset($requestData['userPrompt']) || !isset($requestData['memory'])) {
    xo_respond(['success' => false, 'error' => 'Missing userPrompt or memory in request body'], 400);
}

$userPrompt = $requestData['userPrompt'];
$memory = $requestData['memory'];

// 5. Construct the full prompt (Memory + User Prompt)
$fullPrompt = "USER MEMORY STATE:\n" . json_encode($memory, JSON_PRETTY_PRINT) . "\n\nUSER PROMPT: " . $userPrompt;

$request->setContents([
    new Google\Service\AiPlatform\Content([
        'parts' => [
            new Google\Service\AiPlatform\Part(['text' => $fullPrompt])
        ]
    ])
]);

try {
    // 6. Call the Gemini API
    $response = $service->projects->locations->publishers->models->generateContent(
        "publishers/google/models/$model",
        $request
    );

    // 7. Extract the text response
    $responseText = $response->getGeneratedContent()[0]->getText();
    
    // 8. Find the JSON block in the response (Model can wrap it in markdown)
    $jsonStart = strpos($responseText, '{');
    $jsonEnd = strrpos($responseText, '}');

    if ($jsonStart !== false && $jsonEnd !== false && $jsonEnd > $jsonStart) {
        $jsonBlock = substr($responseText, $jsonStart, $jsonEnd - $jsonStart + 1);
        $modelResponse = json_decode($jsonBlock, true);

        if ($modelResponse !== null && isset($modelResponse['action'])) {
            // Found a valid action object, process it
            if ($modelResponse['action'] === 'respond' && isset($modelResponse['response'])) {
                // Conversational response
                xo_respond([
                    'success' => true,
                    'action'  => 'respond',
                    'response' => $modelResponse['response']
                ]);
            } else {
                // Memory manipulation request, handle it via API endpoint
                require_once 'xo_api.php'; // Requires the memory manipulation logic
                xo_handle_api_request($modelResponse);
            }
        }
    }
    
    // Fallback: If no clean JSON action was detected, send the raw response back.
    xo_respond([
        'success' => true,
        'action' => 'respond',
        'response' => "I received your request, but I couldn't clearly parse the action from the AI's response. Raw output: " . $responseText
    ]);

} catch (Exception $e) {
    // 9. Handle Gemini API errors
    $errorMessage = 'Gemini API Error: ' . $e->getMessage();
    xo_log('chat', 'fatal', ['reason' => 'API_CALL_FAILED', 'error' => $errorMessage]);
    xo_respond(['success' => false, 'error' => $errorMessage], 500);
}