<?php
// add_task.php
// Safely append a new task into xo_memory.json without touching other data.
// Uses impact + effort (+ optional dueDate/projectId) to compute priorityScore/priorityLabel.

ini_set('display_errors', 0);
error_reporting(E_ALL);

require_once __DIR__ . '/xo_common.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    xo_respond(['success' => false, 'error' => 'Method not allowed'], 405);
}

// Parse JSON body
$raw   = file_get_contents('php://input');
$input = json_decode($raw, true);

if (!is_array($input)) {
    xo_log('add_task', 'error', ['reason' => 'invalid_json', 'body' => $raw]);
    xo_respond(['success' => false, 'error' => 'Invalid JSON body'], 400);
}

// Required: title
$title = trim($input['title'] ?? '');
if ($title === '') {
    xo_log('add_task', 'error', ['reason' => 'missing_title', 'input' => $input]);
    xo_respond([
        'success' => false,
        'error'   => 'Missing required field: title',
        'code'    => 'VALIDATION_ERROR'
    ], 400);
}

// Required: impact (1–5), provided by XO based on your description
$impact = $input['impact'] ?? null;
if (is_string($impact) && ctype_digit($impact)) {
    $impact = (int)$impact;
}
if (!is_int($impact) || $impact < 1 || $impact > 5) {
    xo_log('add_task', 'error', ['reason' => 'missing_or_invalid_impact', 'input' => $input]);
    xo_respond([
        'success' => false,
        'error'   => 'Missing or invalid field: impact (expected integer 1–5)',
        'code'    => 'VALIDATION_ERROR'
    ], 400);
}

// Required: effort (1–5), provided by XO based on time estimate
$effort = $input['effort'] ?? null;
if (is_string($effort) && ctype_digit($effort)) {
    $effort = (int)$effort;
}
if (!is_int($effort) || $effort < 1 || $effort > 5) {
    xo_log('add_task', 'error', ['reason' => 'missing_or_invalid_effort', 'input' => $input]);
    xo_respond([
        'success' => false,
        'error'   => 'Missing or invalid field: effort (expected integer 1–5)',
        'code'    => 'VALIDATION_ERROR'
    ], 400);
}

// Optional fields
$status    = $input['status']    ?? 'open';
$dueDate   = $input['dueDate']   ?? null;     // ISO date string or null
$projectId = $input['projectId'] ?? null;     // optional link to a project
$source    = $input['source']    ?? 'XO';
$notes     = $input['notes']     ?? '';

$actor = $source ?: 'XO';

// Normalize projectId to int if provided
if ($projectId !== null) {
    if (is_string($projectId) && ctype_digit($projectId)) {
        $projectId = (int)$projectId;
    }
    if (!is_int($projectId)) {
        // If it's not a clean int, better to drop it than corrupt data
        xo_log('add_task', 'warn', ['reason' => 'invalid_projectId_type', 'value' => $projectId]);
        $projectId = null;
    }
}

// 1) Load existing memory
$memory = xo_load_memory();

// 2) Ensure tasks array exists
if (!isset($memory['data']['tasks']) || !is_array($memory['data']['tasks'])) {
    $memory['data']['tasks'] = [];
}

// 3) Build new task with server-owned id and timestamps
$newId = xo_next_id($memory['data']['tasks']);
$now   = xo_now_iso();

// Compute priority based on impact, effort, and dueDate
list($priorityScore, $priorityLabel) = xo_calculate_task_priority($impact, $effort, $dueDate);

$newTask = [
    'id'            => $newId,
    'title'         => $title,
    'status'        => $status,
    'impact'        => $impact,
    'effort'        => $effort,
    'dueDate'       => $dueDate,
    'projectId'     => $projectId,
    'priorityScore' => $priorityScore,
    'priorityLabel' => $priorityLabel,
    'createdAt'     => $now,
    'updatedAt'     => $now,
    'source'        => $source,
    'notes'         => $notes
];

// 4) Append and write back
$memory['data']['tasks'][] = $newTask;

xo_write_memory($memory, $actor);

// 5) Log and respond
xo_log('add_task', 'success', [
    'id'            => $newId,
    'title'         => $title,
    'impact'        => $impact,
    'effort'        => $effort,
    'priorityScore' => $priorityScore,
    'priorityLabel' => $priorityLabel
]);

xo_respond([
    'success'       => true,
    'action'        => 'add_task',
    'task'          => $newTask,
    'lastUpdatedAt' => $memory['lastUpdatedAt'] ?? null
]);
